use ui;
use core:geometry;
use core:io;
use progvis:net;
use graphics;

dialog ProblemsDlg {
	public Action? action;
	private ProblemState state;
	private ProblemInfo[] problems;

	layout Grid {
		expandCol: 1;
		expandRow: 0;
		ListView list(["Problem", "Author", "Status"]) { row: 0; col: 0; }
		FillBox(300, 200) { row: 0; col: 0; }

		Label initial("Select a problem in the list to the left to start", HAlign:center, VAlign:center) { row: 0; col: 1; }
		Anchor(center) {
			row: 0; col: 1;
			FirstTryPanel firstTry(state) {}
		}
		ExplorePanel explore(state) { row: 0; col: 1; }
		FillBox(400, 200) { row: 0; col: 1; }
	}

	init(Client client, progvis:Settings settings) {
		// Note: This 'state' variable will be available to the layout.
		// This is not the intention. Rather, the layout should have some more structured way to
		// access external resources like this.
		ProblemState state(client, settings);

		init("Problems", Size(1200, 700)) {
			state = state;
		}

		firstTry.visible = false;
		explore.visible = false;
		state.target = this;

		list.onSelect = &this.onListSelect;

		(spawn populateList()).detach();
	}

	// Called by the state object to close the dialog and perform some action.
	void done(Action action) {
		this.action = action;
		close(1);
	}

	private void populateList() {
		unless (response = state.query(ProblemListRequest()) as ProblemListResponse) {
			showMessage(this, "Connection error", "Failed to retrieve list of problems.");
			close(-1);
			return;
		}

		problems = response.data;
		problems.sort();
		for (row in problems) {
			Str status = if (row.attempted) { "Started"; } else { "New"; };
			list.add([row.title, row.author, status]);
		}
	}

	private void onListSelect(Nat id, Bool selected) {
		unless (selected)
			return;

		ProblemInfo info = problems[id];
		state.problem = info.id;

		initial.visible = false;
		if (info.attempted | state.isAdmin) {
			// Show attempted panel.
			firstTry.visible = false;
			explore.visible = true;
		} else {
			// Show "start to solve" panel.
			explore.visible = false;
			firstTry.visible = true;
		}
	}

}

/**
 * Callback for the problem dialog.
 */
package class ProblemState on Ui {
	public Client client;
	private progvis:Settings settings;
	private Int currentProblem;
	private Array<Fn<void, Int>> callbacks;

	ProblemsDlg? target;

	init(Client client, progvis:Settings settings) {
		init {
			client = client;
			settings = settings;
			currentProblem = -1;
		}
	}

	Bool isAdmin() { client.isAdmin; }

	Response query(Request request) {
		client.query(request);
	}

	void message(Str message) {
		if (target) {
			showMessage(target, "Error", message);
		}
	}

	void add(Fn<void, Int> fn) {
		callbacks << fn;
	}

	assign problem(Int problemId) {
		if (problemId != currentProblem) {
			currentProblem = problemId;
			// Trigger callbacks.
			(spawn reloadProblem()).detach();
		}
	}

	Int problem() { currentProblem; }

	void reloadProblem() {
		// Trigger callbacks.
		Int problemId = currentProblem;
		for (x in callbacks) {
			try {
				x.call(problemId);
			} catch (Exception e) {
				print("Unhandled exception: ${e}");
			}
		}
	}

	void solveProblem(Problem problem) {
		if (problem.error.unknown()) {
			message("TODO: We need to check for errors locally!");
		} else if (e = problem.error.error) {
			if (progvis:program:reportedInteractively(e)) {
				done(DebugAction(settings, problem));
			} else {
				done(ImplAction(settings, problem));
			}
		} else {
			done(TestAction(settings, problem));
		}
	}

	void done(Action action) {
		if (target) {
			target.done(action);
		}
	}

	void newImpl(Problem problem) {
		done(ImplAction(settings, problem));
	}

	void newTest(Problem problem) {
		done(TestAction(settings, problem));
	}

}

window FirstTryPanel {
	layout Grid {
		Label("You have not solved this problem before.", HAlign:center) {}
		nextLine;
		Label("Click the button below to start solving it!", HAlign:center) {}
		nextLine;
		FillBox(10, 20) {}
		nextLine;
		Anchor(center) { Button ok("Solve") {} }
	}

	private ProblemState state;

	init(ProblemState state) {
		init() {
			state = state;
		}

		ok.onClick = &this.onOk;
	}

	private void onOk() {
		ok.enabled = false;
		(spawn fetchProblem()).detach();
	}

	private void fetchProblem() {
		try {
			unless (response = state.query(ProblemRequest(state.problem)) as ProblemResponse) {
				state.message("Failed to fetch the problem.");
				ok.enabled = true;
				return;
			}

			state.solveProblem(response.problem);
		} catch (Exception e) {
			state.message("Failed to fetch the problem: " + e.message());
			ok.enabled = true;
		}
	}
}

class ExplorePanel extends TabView {
	init(ProblemState state) {
		init() {}

		add("Overview", CodeOverviewPanel(state));
		add("Improve my implementation", ExploreTestsPanel(state));
		add("Improve my test", ExploreImplPanel(state));
	}
}

window CodeOverviewPanel {
	layout Grid {
		expandRow: 2;
		expandCol: 0;
		expandCol: 1;
		normalizeCols;

		Label implLbl("Your implementation") {}
		Label testLbl("Your tests:") {}
		nextLine;
		CodeView implCode("Loading...") { rowspan: 2; }
		ListView testList(["Version", "Status"]) {}
		FillBox(10, 150) { col: 1; }
		nextLine;
		CodeView testCode("Select a version above") { col: 1; }
		nextLine;
		CheckProgress checker { col: 0; colspan: 2; }
		Grid { col: 0; colspan: 2;
			expandCol: 0;
			expandRow: 0;
			FillBox(1, 1) {}
			Button newImpl("Admin: New impl.") {}
			Button newTest("Admin: New test") {}
			Button solve("Investigate") {}
		}
	}

	ProblemState state;
	private CurrentStateResponse? showing;
	private Nat showingId;

	init(ProblemState state) {
		init() {
			state = state;
		}

		if (state.isAdmin) {
			newImpl.onClick = &this.onNewImpl;
			newTest.onClick = &this.onNewTest;
		} else {
			newImpl.visible = false;
			newTest.visible = false;
		}

		solve.onClick = &this.onSolve;

		state.add(&this.onInfo);
		testList.onSelect = &this.onTestSelected;
	}

	private void onInfo(Int id) {
		unless (response = state.query(CurrentStateRequest(id)) as CurrentStateResponse)
			return;

		showing = response;
		showingId = 0;

		if (response.implId < 0)
			implLbl.text = "Original implementation:";
		else
			implLbl.text = "Version ${response.implVersion} of your implementation:";

		implCode.show(response.impl);

		testList.clear();
		for (id, error in response.errors) {
			Str id = if (id == 0) { "Original"; } else { id.toS; };
			Str err = if (error.unknown) {
				"Not checked yet";
			} else if (error = error.error) {
				"Implementation fails with: " + error;
			} else {
				"Implementation passes this test";
			};
			testList.add([id, err]);
		}
		testList.selection = testList.count - 1;
	}

	private void onTestSelected(Nat id, Bool selected) {
		if (!selected)
			return;
		unless (showing)
			return;
		unless (id < showing.tests.count)
			return;

		showingId = id;

		testCode.show(showing.tests[id]);
		var error = showing.errors[id];

		if (error.unknown) {
			solve.text = "Check";
		} else if (error.success) {
			solve.text = "Improve test";
		} else {
			solve.text = "Debug";
		}
		resized();
	}

	private void onSolve() {
		unless (showing)
			return;

		var error = showing.errors[showingId];
		if (error.unknown) {
			solve.visible = false;
			newImpl.visible = false;
			newTest.visible = false;
			checker.check(state.client, showing.id, [showing.implId], ImprovementType:implementation, &this.checkDone);
		} else {
			state.solveProblem(showing.problemFor(showingId));
		}
	}

	private void checkDone() {
		solve.visible = true;
		if (state.isAdmin) {
			newImpl.visible = true;
			newTest.visible = true;
		}

		if (showing) {
			if (state.problem == showing.id)
				state.reloadProblem();
		}
	}

	private void onNewImpl() {
		unless (showing)
			return;

		state.newImpl(showing.problemFor(showingId));
	}

	private void onNewTest() {
		unless (showing)
			return;

		state.newTest(showing.problemFor(showingId));
	}

}

window ExploreImprovementPanel {
	layout Grid {
		expandRow: 1;
		expandCol: 0, 45;
		expandCol: 1, 60;
		normalizeCols;

		Label lblAvail("Available") {}
		Label lblPreview("Preview of the selected:") {}
		nextLine;
		ListView improvementsView(["Author", "Version", "Status"]) {}
		CodeView code {}
		nextLine;
		CheckProgress checker { row: 2; col: 0; colspan: 2; }
		Grid { row: 2; col: 1;
			expandCol: 0;
			expandCol: 1;
			expandRow: 0;
			normalizeCols;
			Button checkAll("Check all") {}
			Button solve("Check") {}
		}
	}

	ProblemState state;
	protected ImprovementType type;
	protected Int problemId;
	protected Improvement[] improvements;
	protected Nat improvementId;

	init(ProblemState state, ImprovementType type) {
		init {
			state = state;
			type = type;
		}

		if (type == ImprovementType:test) {
			lblAvail.text = "Available tests:";
			lblPreview.text = "Preview of the selected test:";
		} else {
			lblAvail.text = "Available implementations:";
			lblPreview.text = "Preview of the selected implementation:";
		}

		improvementsView.onSelect = &this.onSelect;
		checkAll.onClick = &this.onCheckAll;
		solve.onClick = &this.onSolve;
		solve.enabled = false;
		state.add(&this.onInfo);
	}

	private void onInfo(Int id) {
		// If we are reloading the same problem, try to preserve the location of the marker.
		Bool update = id == problemId;

		if (!update) {
			problemId = id;
			code.message("Select an item to the left");
			improvementsView.clear();
			improvements.clear();
			improvementId = 0;
		}

		unless (response = state.query(ImprovementsRequest(problemId, type)) as ImprovementsResponse)
			return;

		improvements = response.data;

		if (update) {
			// These lists can not really shrink, so this is enough.
			for (row, i in improvements) {
				if (row < improvementsView.count)
					improvementsView.update(row, [i.author, i.version.toS, status(i.error, i.errorVersion)]);
				else
					improvementsView.add([i.author, i.version.toS, status(i.error, i.errorVersion)]);
			}
		} else {
			improvementId = improvements.count;

			for (i in improvements) {
				improvementsView.add([i.author, i.version.toS, status(i.error, i.errorVersion)]);
			}
		}

		// Update buttons etc.
		updateCode(improvementId);
	}

	private void onSelect(Nat id, Bool selected) {
		if (!selected)
			return;

		(spawn updateCode(id)).detach();
	}

	private void updateCode(Nat id) {
		improvementId = id;
		if (id >= improvements.count) {
			solve.text = "Pick an item";
			solve.enabled = false;
			return;
		}

		unless (response = state.query(ImprovementCodeRequest(improvements[id].id, type)) as CodeResponse)
			return;

		code.show(response.code);
		if (solveMsg = solveMessage(improvements[id])) {
			solve.text = solveMsg;
			solve.enabled = true;
		} else {
			solve.text = "Pick another item";
			solve.enabled = false;
		}
	}

	private void onSolve() {
		var improvement = improvements[improvementId];
		if (improvement.error.unknown()) {
			check([improvement.id]);
		} else if (response = state.query(ImprovedProblemRequest(problemId, improvement.id, type)) as ProblemResponse) {
			// TODO: When trying to improve a test, we should probably ask which test to use as a
			// starting point rather than just picking the latest one. This behavior is fine for
			// now, however.
			state.solveProblem(response.problem);
		}
	}

	private void onCheckAll() {
		Int[] toCheck;
		for (i in improvements)
			if (i.error.unknown)
				toCheck << i.id;

		if (toCheck.any)
			check(toCheck);
	}

	private void check(Int[] check) {
		checkAll.visible = false;
		solve.visible = false;
		checker.check(state.client, problemId, check, type, &this.checkDone);
	}

	private void checkDone() {
		checkAll.visible = true;
		solve.visible = true;

		if (state.problem == problemId) {
			state.reloadProblem();
		}
	}

	// Message to show on the solve button.
	protected Str? solveMessage(Improvement improvement) : abstract;

	// What to show as "status"?
	protected Str status(Error error, Int errorVersion) : abstract;
}

class ExploreTestsPanel extends ExploreImprovementPanel {
	init(ProblemState state) {
		init(state, ImprovementType:test) {}
	}

	protected Str? solveMessage(Improvement improvement) : override {
		if (improvement.error.unknown) {
			return "Check";
		} else if (improvement.error.success) {
			return null;
		} else {
			return "Improve my implementation";
		}
	}

	protected Str status(Error error, Int errorVersion) : override {
		if (error.unknown) {
			return "Not checked yet";
		} else if (e = error.error) {
			return "Finds error: " + e;
		} else {
			return "Does not find errors";
		}
	}
}

class ExploreImplPanel extends ExploreImprovementPanel {
	init(ProblemState state) {
		init(state, ImprovementType:implementation) {}
	}

	protected Str? solveMessage(Improvement improvement) : override {
		if (improvement.error.unknown) {
			return "Check";
		} else if (improvement.error.success) {
			return "Improve my test";
		} else {
			return null;
		}
	}

	protected Str status(Error error, Int errorVersion) : override {
		if (error.unknown) {
			return "Not checked yet";
		} else if (e = error.error) {
			return "Version ${errorVersion} of your test found: ${e}";
		} else {
			return "No errors found by any version of your test";
		}
	}
}

window CheckProgress {
	layout without border Grid {
		expandCol: 1;
		expandRow: 1;
		Label msg("Checking...", VAlign:center) {}
		Progress progress {}
		Button cancel("Cancel") {}
	}

	init() {
		init() {}

		visible = false;
		cancel.onClick = &this.onCancel;
	}

	private CheckCallback callback;
	private Bool cancelled;

	void check(Client client, Int problem, Int[] improvements, ImprovementType type, Fn<void> doneCb) {
		if (visible)
			return;

		visible = true;
		cancelled = false;
		callback = CheckCallback();

		msg.text = "Initializing...";
		progress.wait();
		resized();

		(spawn startCheck(client, problem, improvements, type, doneCb)).detach();
	}

	private void startCheck(Client client, Int problem, Int[] improvements, ImprovementType type, Fn<void> doneCb) {
		// Figure out what to do:
		CodeTriplet[] toCheck;
		for (imp in improvements) {
			unless (response = client.query(MissingChecksRequest(problem, imp, type)) as MissingChecksResponse)
				continue;

			toCheck.append(response.missing);
		}

		// Do it:
		try {
			Nat count = toCheck.count;
			for (id, triplet in toCheck) {
				if (cancelled)
					break;

				if (count > 1) {
					progress.progress = id.float / count.float;
					msg.text = "Checking (${id + 1}/${count})...";
					resized();
				}

				doCheck(client, triplet);
			}
		} catch (Exception e) {
			print("Internal error: " + e.message);
		}

		visible = false;
		doneCb.call();
	}

	private void doCheck(Client client, CodeTriplet triplet) {
		unless (r = client.query(ProblemRequest(triplet)) as ProblemResponse)
			return;

		var problem = r.problem;

		MemoryProtocol memory;
		Url test = problem.test.put("test", memory);
		Url impl = problem.impl.put("impl", memory);

		Error result = Error:unknown();
		try {
			if (error = problem.checks.check([test, impl], callback)) {
				result = Error:error(error.error.type);
			} else {
				result = Error:success();
			}
		} catch (Exception e) {
			print("Failed to check a problem: " + e.message);
			return;
		}

		client.query(StoreCheckResult(problem.id, problem.implId, problem.testId, result));
	}

	private void onCancel() {
		cancelled = true;
		callback.setCancel();
	}
}

class CheckCallback extends progvis:check:Progress {
	Bool cancel;
	void setCancel() {
		cancel = true;
	}
	void progress(Nat depth, Nat states, Nat edges, Nat unexplored) : override {
		if (cancel)
			throw progvis:program:ExitError();
	}
}
